/**
 * BriefFlow Background Service Worker
 * Manages brief sessions, API communication, and screenshot capture
 */

const API_BASE = 'https://briefflow-api.phil-76c.workers.dev';
const ACTIVATION_HOST = 'briefflow-activate.pages.dev';

// ============================================
// SESSION MANAGEMENT
// ============================================

async function getSession() {
  const data = await chrome.storage.local.get(['active_brief', 'messages']);
  return data.active_brief ? { brief: data.active_brief, messages: data.messages || [] } : null;
}

async function setSession(brief, messages = []) {
  await chrome.storage.local.set({ active_brief: brief, messages });
}

async function clearSession() {
  await chrome.storage.local.remove(['active_brief', 'messages']);
}

async function updateMessages(messages) {
  await chrome.storage.local.set({ messages });
}

async function updateBriefState(updates) {
  const session = await getSession();
  if (session) {
    const updatedBrief = { ...session.brief, ...updates };
    await chrome.storage.local.set({ active_brief: updatedBrief });
  }
}

// ============================================
// API COMMUNICATION
// ============================================

async function activateBrief(token, mode = null) {
  try {
    // Add mode parameter if in complete mode
    let url = `${API_BASE}/api/c/${token}`;
    if (mode === 'complete') {
      url += '?mode=complete';
    }
    
    const response = await fetch(url);
    
    if (!response.ok) {
      const error = await response.json();
      return { success: false, error: error.error || 'Failed to activate brief' };
    }
    
    const data = await response.json();
    
    const brief = {
      token,
      client_name: data.client_name,
      status: data.status,
      progress: data.progress,
      starting_url: data.starting_url,
      preview: data.preview || false,
      force_complete: data.force_complete || false,
    };
    
    await setSession(brief, data.messages || []);
    
    return { success: true, brief, messages: data.messages || [] };
  } catch (error) {
    console.error('Activation error:', error);
    return { success: false, error: 'Network error' };
  }
}

async function sendMessageToAPI(content, urlContext = null, screenshotUrl = null, type = 'message') {
  const session = await getSession();
  if (!session) {
    return { success: false, error: 'No active session' };
  }
  
  try {
    const response = await fetch(`${API_BASE}/api/c/${session.brief.token}`, {
      method: 'POST',
      headers: { 'Content-Type': 'application/json' },
      body: JSON.stringify({
        content,
        url_context: urlContext,
        screenshot_url: screenshotUrl,
        type,
        force_complete: session.brief.force_complete || false,
      }),
    });
    
    if (!response.ok) {
      const error = await response.json();
      return { success: false, error: error.error || 'Failed to send message' };
    }
    
    const data = await response.json();
    
    // Update local state
    const newMessages = [
      ...session.messages,
      { role: type === 'navigation' ? 'system' : 'user', content, url_context: urlContext, screenshot_url: screenshotUrl },
      { role: 'assistant', content: data.message, display_hints: { celebration: data.celebration, progress: data.progress } },
    ];
    
    await updateMessages(newMessages);
    await updateBriefState({
      progress: data.progress,
      status: data.status === 'complete' ? 'complete' : session.brief.status,
    });
    
    return { success: true, data };
  } catch (error) {
    console.error('Send message error:', error);
    return { success: false, error: 'Network error' };
  }
}

async function uploadScreenshot(dataUrl) {
  const session = await getSession();
  if (!session) {
    return { success: false, error: 'No active session' };
  }
  
  try {
    // Convert data URL to blob
    const response = await fetch(dataUrl);
    const blob = await response.blob();
    
    const formData = new FormData();
    formData.append('screenshot', blob, 'screenshot.png');
    
    const uploadResponse = await fetch(`${API_BASE}/api/c/${session.brief.token}/screenshot`, {
      method: 'POST',
      body: formData,
    });
    
    if (!uploadResponse.ok) {
      return { success: false, error: 'Upload failed' };
    }
    
    const data = await uploadResponse.json();
    return { success: true, screenshot_url: data.screenshot_url };
  } catch (error) {
    console.error('Upload error:', error);
    return { success: false, error: 'Upload failed' };
  }
}

// ============================================
// SCREENSHOT CAPTURE
// ============================================

async function captureCurrentTab() {
  try {
    const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
    if (!tab) {
      return { success: false, error: 'No active tab' };
    }
    
    const dataUrl = await chrome.tabs.captureVisibleTab(null, { format: 'png' });
    return { success: true, dataUrl, tabUrl: tab.url };
  } catch (error) {
    console.error('Capture error:', error);
    return { success: false, error: 'Failed to capture screenshot' };
  }
}

// ============================================
// URL TRACKING
// ============================================

let lastUrl = null;

chrome.tabs.onUpdated.addListener(async (tabId, changeInfo, tab) => {
  if (changeInfo.url && tab.active) {
    const session = await getSession();
    if (session && session.brief.status !== 'complete') {
      // Only track if URL actually changed
      if (changeInfo.url !== lastUrl && !changeInfo.url.includes(ACTIVATION_HOST)) {
        lastUrl = changeInfo.url;
        // Notify content script about URL change
        try {
          await chrome.tabs.sendMessage(tabId, {
            type: 'URL_CHANGED',
            url: changeInfo.url,
          });
        } catch (e) {
          // Tab might not have content script yet
        }
      }
    }
  }
});

// ============================================
// MESSAGE HANDLERS
// ============================================

chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  // Handle async responses
  handleMessage(message, sender).then(sendResponse);
  return true; // Keep channel open for async response
});

async function handleMessage(message, sender) {
  switch (message.type) {
    case 'ACTIVATE_BRIEF':
      return await activateBrief(message.token, message.mode);
    
    case 'GET_SESSION':
      return await getSession();
    
    case 'SEND_MESSAGE':
      return await sendMessageToAPI(message.content, message.url_context);
    
    case 'CAPTURE_SCREENSHOT':
      const capture = await captureCurrentTab();
      if (!capture.success) {
        return capture;
      }
      
      // Upload screenshot
      const upload = await uploadScreenshot(capture.dataUrl);
      if (!upload.success) {
        return upload;
      }
      
      // Send message with screenshot
      return await sendMessageToAPI(
        message.caption || 'Here\'s what I see',
        capture.tabUrl,
        upload.screenshot_url
      );
    
    case 'LOG_NAVIGATION':
      return await sendMessageToAPI(
        `Navigated to ${message.url}`,
        message.url,
        null,
        'navigation'
      );
    
    case 'CLEAR_SESSION':
      await clearSession();
      return { success: true };
    
    case 'GET_INITIAL_MESSAGE':
      // Generate initial greeting if no messages yet
      const session = await getSession();
      if (session && session.messages.length === 0) {
        return await sendMessageToAPI('Hello, I\'m ready to start the brief.', null, null, 'message');
      }
      return { success: true, data: null };
    
    default:
      return { success: false, error: 'Unknown message type' };
  }
}

// ============================================
// EXTENSION ICON CLICK
// ============================================

chrome.action.onClicked.addListener(async (tab) => {
  // Toggle widget visibility
  try {
    await chrome.tabs.sendMessage(tab.id, { type: 'TOGGLE_WIDGET' });
  } catch (e) {
    // Content script not loaded yet
  }
});

// ============================================
// INSTALLATION
// ============================================

chrome.runtime.onInstalled.addListener(() => {
  console.log('BriefFlow extension installed');
});
