/**
 * BriefFlow Content Script
 * Injects and manages the chat widget on all pages
 */

const ACTIVATION_HOST = 'briefflow-activate.pages.dev';

// ============================================
// WIDGET INJECTION
// ============================================

let widgetContainer = null;
let isWidgetVisible = false;
let currentSession = null;

function createWidget() {
  if (widgetContainer) return;
  
  widgetContainer = document.createElement('div');
  widgetContainer.id = 'briefflow-widget-container';
  widgetContainer.innerHTML = `
    <div id="briefflow-widget" class="briefflow-widget briefflow-hidden">
      <div class="briefflow-header">
        <div class="briefflow-title">
          <span class="briefflow-logo">✦</span>
          BriefFlow
          <span id="briefflow-preview-badge" class="briefflow-preview-badge briefflow-hidden">👁 Preview</span>
        </div>
        <div class="briefflow-controls">
          <button id="briefflow-minimize" class="briefflow-control-btn" title="Minimize">─</button>
          <button id="briefflow-close" class="briefflow-control-btn" title="Close">✕</button>
        </div>
      </div>
      
      <div id="briefflow-messages" class="briefflow-messages">
        <!-- Messages will be inserted here -->
      </div>
      
      <div id="briefflow-complete" class="briefflow-complete briefflow-hidden">
        <div class="briefflow-complete-icon">🎉</div>
        <div class="briefflow-complete-title">All done!</div>
        <div class="briefflow-complete-text">Thanks for your time. The CCM team will be in touch soon.</div>
      </div>
      
      <div id="briefflow-input-area" class="briefflow-input-area">
        <div class="briefflow-input-wrapper">
          <input type="text" id="briefflow-input" placeholder="Type here..." autocomplete="off">
          <button id="briefflow-screenshot" class="briefflow-icon-btn" title="Take screenshot">📷</button>
          <button id="briefflow-send" class="briefflow-icon-btn" title="Send">➤</button>
        </div>
      </div>
      
      <div class="briefflow-progress">
        <div id="briefflow-progress-bar" class="briefflow-progress-bar" style="width: 0%"></div>
      </div>
      
      <div id="briefflow-saved" class="briefflow-saved briefflow-hidden">✓ Saved</div>
    </div>
    
    <button id="briefflow-fab" class="briefflow-fab briefflow-hidden">
      <span class="briefflow-fab-icon">✦</span>
    </button>
    
    <div id="briefflow-confetti" class="briefflow-confetti"></div>
  `;
  
  document.body.appendChild(widgetContainer);
  attachEventListeners();
}

function attachEventListeners() {
  // Minimize button
  document.getElementById('briefflow-minimize').addEventListener('click', minimizeWidget);
  
  // Close button
  document.getElementById('briefflow-close').addEventListener('click', hideWidget);
  
  // FAB to restore
  document.getElementById('briefflow-fab').addEventListener('click', showWidget);
  
  // Send message
  document.getElementById('briefflow-send').addEventListener('click', handleSendMessage);
  document.getElementById('briefflow-input').addEventListener('keypress', (e) => {
    if (e.key === 'Enter' && !e.shiftKey) {
      e.preventDefault();
      handleSendMessage();
    }
  });
  
  // Screenshot button
  document.getElementById('briefflow-screenshot').addEventListener('click', handleScreenshot);
}

// ============================================
// WIDGET STATE MANAGEMENT
// ============================================

function showWidget() {
  const widget = document.getElementById('briefflow-widget');
  const fab = document.getElementById('briefflow-fab');
  
  widget.classList.remove('briefflow-hidden', 'briefflow-minimized');
  fab.classList.add('briefflow-hidden');
  isWidgetVisible = true;
}

function hideWidget() {
  const widget = document.getElementById('briefflow-widget');
  const fab = document.getElementById('briefflow-fab');
  
  widget.classList.add('briefflow-hidden');
  fab.classList.add('briefflow-hidden');
  isWidgetVisible = false;
}

function minimizeWidget() {
  const widget = document.getElementById('briefflow-widget');
  const fab = document.getElementById('briefflow-fab');
  
  widget.classList.add('briefflow-hidden');
  fab.classList.remove('briefflow-hidden');
  isWidgetVisible = false;
}

function setPreviewMode(isPreview) {
  const badge = document.getElementById('briefflow-preview-badge');
  if (isPreview) {
    badge.classList.remove('briefflow-hidden');
  } else {
    badge.classList.add('briefflow-hidden');
  }
}

function setProgress(percent) {
  const bar = document.getElementById('briefflow-progress-bar');
  bar.style.width = `${percent}%`;
}

function showComplete() {
  const messages = document.getElementById('briefflow-messages');
  const complete = document.getElementById('briefflow-complete');
  const inputArea = document.getElementById('briefflow-input-area');
  
  messages.classList.add('briefflow-hidden');
  inputArea.classList.add('briefflow-hidden');
  complete.classList.remove('briefflow-hidden');
  
  triggerCelebration('confetti');
}

function showSavedIndicator() {
  const saved = document.getElementById('briefflow-saved');
  saved.classList.remove('briefflow-hidden');
  setTimeout(() => {
    saved.classList.add('briefflow-hidden');
  }, 1500);
}

// ============================================
// MESSAGES
// ============================================

function renderMessages(messages) {
  const container = document.getElementById('briefflow-messages');
  container.innerHTML = '';
  
  messages.forEach(msg => {
    if (msg.role === 'system') return; // Don't show system messages
    
    const div = document.createElement('div');
    div.className = `briefflow-message briefflow-${msg.role}`;
    
    let content = msg.content;
    
    // Add screenshot thumbnail if present
    if (msg.screenshot_url) {
      content = `<div class="briefflow-screenshot-thumb"><img src="${msg.screenshot_url}" alt="Screenshot"></div>` + content;
    }
    
    div.innerHTML = content;
    container.appendChild(div);
  });
  
  // Scroll to bottom
  container.scrollTop = container.scrollHeight;
}

function addMessage(role, content, screenshotUrl = null) {
  const container = document.getElementById('briefflow-messages');
  
  const div = document.createElement('div');
  div.className = `briefflow-message briefflow-${role}`;
  
  let html = content;
  if (screenshotUrl) {
    html = `<div class="briefflow-screenshot-thumb"><img src="${screenshotUrl}" alt="Screenshot"></div>` + html;
  }
  
  div.innerHTML = html;
  container.appendChild(div);
  container.scrollTop = container.scrollHeight;
}

function addLoadingMessage() {
  const container = document.getElementById('briefflow-messages');
  
  const div = document.createElement('div');
  div.className = 'briefflow-message briefflow-assistant briefflow-loading';
  div.id = 'briefflow-loading-msg';
  div.innerHTML = '<span class="briefflow-dots"><span>.</span><span>.</span><span>.</span></span>';
  
  container.appendChild(div);
  container.scrollTop = container.scrollHeight;
}

function removeLoadingMessage() {
  const loading = document.getElementById('briefflow-loading-msg');
  if (loading) {
    loading.remove();
  }
}

// ============================================
// USER ACTIONS
// ============================================

async function handleSendMessage() {
  const input = document.getElementById('briefflow-input');
  const content = input.value.trim();
  
  if (!content) return;
  
  input.value = '';
  input.disabled = true;
  
  // Add user message immediately
  addMessage('user', content);
  addLoadingMessage();
  
  // Send to background
  const response = await chrome.runtime.sendMessage({
    type: 'SEND_MESSAGE',
    content,
    url_context: window.location.href,
  });
  
  removeLoadingMessage();
  input.disabled = false;
  input.focus();
  
  if (response.success) {
    addMessage('assistant', response.data.message);
    setProgress(response.data.progress);
    showSavedIndicator();
    
    if (response.data.celebration && response.data.celebration !== 'none') {
      triggerCelebration(response.data.celebration);
    }
    
    if (response.data.status === 'complete') {
      showComplete();
    }
  } else {
    addMessage('assistant', 'Sorry, something went wrong. Please try again.');
  }
}

async function handleScreenshot() {
  const btn = document.getElementById('briefflow-screenshot');
  btn.disabled = true;
  btn.textContent = '⏳';
  
  addLoadingMessage();
  
  const response = await chrome.runtime.sendMessage({
    type: 'CAPTURE_SCREENSHOT',
    caption: 'Here\'s what I see',
  });
  
  removeLoadingMessage();
  btn.disabled = false;
  btn.textContent = '📷';
  
  if (response.success) {
    addMessage('user', 'Here\'s what I see', response.data?.screenshot_url);
    addMessage('assistant', response.data.message);
    setProgress(response.data.progress);
    showSavedIndicator();
    
    if (response.data.celebration && response.data.celebration !== 'none') {
      triggerCelebration(response.data.celebration);
    }
    
    if (response.data.status === 'complete') {
      showComplete();
    }
  } else {
    addMessage('assistant', 'Failed to capture screenshot. Please try again.');
  }
}

// ============================================
// CELEBRATIONS
// ============================================

function triggerCelebration(type) {
  const widget = document.getElementById('briefflow-widget');
  
  switch (type) {
    case 'confetti':
      launchConfetti();
      break;
    case 'sparkle':
      widget.classList.add('briefflow-sparkle');
      setTimeout(() => widget.classList.remove('briefflow-sparkle'), 1000);
      break;
    case 'checkmark':
      showCheckmark();
      break;
    case 'milestone':
      widget.classList.add('briefflow-milestone');
      setTimeout(() => widget.classList.remove('briefflow-milestone'), 1500);
      break;
  }
}

function launchConfetti() {
  const container = document.getElementById('briefflow-confetti');
  container.innerHTML = '';
  
  const colors = ['#ff6b6b', '#4ecdc4', '#45b7d1', '#96e6a1', '#dda0dd', '#f7dc6f'];
  
  for (let i = 0; i < 100; i++) {
    const confetto = document.createElement('div');
    confetto.className = 'briefflow-confetto';
    confetto.style.left = Math.random() * 100 + '%';
    confetto.style.backgroundColor = colors[Math.floor(Math.random() * colors.length)];
    confetto.style.animationDelay = Math.random() * 0.5 + 's';
    confetto.style.animationDuration = (Math.random() * 1 + 2) + 's';
    container.appendChild(confetto);
  }
  
  setTimeout(() => {
    container.innerHTML = '';
  }, 3000);
}

function showCheckmark() {
  const messages = document.getElementById('briefflow-messages');
  const check = document.createElement('div');
  check.className = 'briefflow-checkmark-anim';
  check.innerHTML = '✓';
  messages.appendChild(check);
  
  setTimeout(() => check.remove(), 1500);
}

// ============================================
// ACTIVATION PAGE HANDLING
// ============================================

async function checkForActivation() {
  const url = new URL(window.location.href);
  
  // Check if we're on the activation page (root path or /start)
  if (url.hostname === ACTIVATION_HOST && (url.pathname === '/' || url.pathname === '/start' || url.pathname === '')) {
    const token = url.searchParams.get('token');
    const mode = url.searchParams.get('mode'); // 'complete' or null for preview
    
    if (token) {
      // Notify activation page that extension is present
      window.postMessage({ type: 'BRIEFFLOW_EXTENSION_READY' }, '*');
      
      // Attempt to activate the brief (pass mode to background)
      const response = await chrome.runtime.sendMessage({
        type: 'ACTIVATE_BRIEF',
        token,
        mode, // Will be 'complete' or null
      });
      
      if (response.success) {
        currentSession = response;
        
        // Notify activation page of success
        window.postMessage({ 
          type: 'BRIEFFLOW_ACTIVATED',
          starting_url: response.brief.starting_url,
        }, '*');
      } else {
        window.postMessage({ 
          type: 'BRIEFFLOW_ACTIVATION_FAILED',
          error: response.error,
        }, '*');
      }
    }
  }
}

// ============================================
// INITIALIZATION
// ============================================

async function initialize() {
  // Check if we're on the activation page
  const url = new URL(window.location.href);
  if (url.hostname === ACTIVATION_HOST) {
    await checkForActivation();
    return;
  }
  
  // Check for existing session
  const session = await chrome.runtime.sendMessage({ type: 'GET_SESSION' });
  
  if (session && session.brief && session.brief.status !== 'complete' && session.brief.status !== 'revoked') {
    currentSession = session;
    
    // Create and show widget
    createWidget();
    renderMessages(session.messages || []);
    setProgress(session.brief.progress || 0);
    setPreviewMode(session.brief.preview || false);
    showWidget();
    
    // If no messages yet, get initial greeting
    if (!session.messages || session.messages.length === 0) {
      addLoadingMessage();
      const response = await chrome.runtime.sendMessage({ type: 'GET_INITIAL_MESSAGE' });
      removeLoadingMessage();
      
      if (response.success && response.data) {
        addMessage('assistant', response.data.message);
        setProgress(response.data.progress);
      }
    }
  }
}

// ============================================
// MESSAGE LISTENERS
// ============================================

// Listen for messages from background script
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  switch (message.type) {
    case 'TOGGLE_WIDGET':
      if (!widgetContainer) {
        createWidget();
      }
      if (isWidgetVisible) {
        minimizeWidget();
      } else {
        showWidget();
      }
      sendResponse({ success: true });
      break;
    
    case 'URL_CHANGED':
      // Log navigation if we have an active session
      if (currentSession && currentSession.brief) {
        chrome.runtime.sendMessage({
          type: 'LOG_NAVIGATION',
          url: message.url,
        });
      }
      sendResponse({ success: true });
      break;
  }
  return true;
});

// Listen for ping from activation page
window.addEventListener('message', (event) => {
  if (event.data.type === 'BRIEFFLOW_PING') {
    window.postMessage({ type: 'BRIEFFLOW_PONG' }, '*');
  }
});

// Start
initialize();
